function simulate_swing_arm_real_data()
    % SIMULATE_SWING_ARM_REAL_DATA
    % Simulates the swing arm movement with real-world data inputs.

    %% Real Data Inputs
    fieldSize = 1000;          % Field size (1000x1000 meters)
    pivotCenter = [500, 500];  % Center pivot location (meters)
    pivotRadius = 500;         % Radius of last tower (meters)
    armLength = 207.1;         % Length of the swing arm (meters)
    pivotSpeed = 1.0;          % Pivot speed (m/s)

    % Corner data (start and end angles in degrees)
    corners = struct( ...
        'q1', struct('start', 10, 'end', 80),  ...
        'q2', struct('start', 100, 'end', 170), ...
        'q3', struct('start', 190, 'end', 260), ...
        'q4', struct('start', 280, 'end', 350) ...
    );

    % Simulation parameters
    totalTime = 600;  % Total time for full rotation (seconds)
    dt = 0.1;         % Time step (seconds)
    numSteps = totalTime / dt;
    theta_deg = linspace(0, 360, numSteps);  % Pivot angles (degrees)
    
    %% Preallocate for performance
    xTower = zeros(1, numSteps);
    yTower = zeros(1, numSteps);
    alpha_deg = zeros(1, numSteps);  % Swing arm angle
    xTip = zeros(1, numSteps);
    yTip = zeros(1, numSteps);

    %% Main Simulation Loop
    for k = 1:numSteps
        theta = theta_deg(k);  % Current pivot angle (degrees)

        % Compute tower position
        xTower(k) = pivotCenter(1) + pivotRadius * cosd(theta);
        yTower(k) = pivotCenter(2) + pivotRadius * sind(theta);

        % Compute alpha angle
        alpha_deg(k) = computeAlphaInCorners(theta, corners);

        % Compute tip direction and position
        tipDir = theta + (alpha_deg(k) - 90);
        xTip(k) = xTower(k) + armLength * cosd(tipDir);
        yTip(k) = yTower(k) + armLength * sind(tipDir);

        % Ensure tip stays within field boundaries
        xTip(k) = min(max(xTip(k), 0), fieldSize);
        yTip(k) = min(max(yTip(k), 0), fieldSize);
    end

    %% Visualization
    visualizeSwingArm(pivotCenter, fieldSize, theta_deg, xTower, yTower, xTip, yTip, alpha_deg);
end

%% Function to Compute Alpha Angle in Corners
function alpha = computeAlphaInCorners(theta, corners)
    % Default alpha is 90 (fully retracted)
    alpha = 90;

    % Check which corner the pivot is in and adjust alpha
    if theta >= 0 && theta < 90
        alpha = rampAlpha(theta, corners.q1.start, corners.q1.end);
    elseif theta >= 90 && theta < 180
        alpha = rampAlpha(theta, corners.q2.start, corners.q2.end);
    elseif theta >= 180 && theta < 270
        alpha = rampAlpha(theta, corners.q3.start, corners.q3.end);
    elseif theta >= 270 && theta < 360
        alpha = rampAlpha(theta, corners.q4.start, corners.q4.end);
    end
end

%% Function to Ramp Alpha in Corners
function alpha = rampAlpha(theta, startAngle, endAngle)
    % Compute alpha in [90,180] range
    if theta < startAngle || theta > endAngle
        alpha = 90;  % Fully retracted outside corner
    else
        midAngle = (startAngle + endAngle) / 2;
        if theta <= midAngle
            % Ramp up: 90 -> 180
            alpha = 90 + (180 - 90) * ((theta - startAngle) / (midAngle - startAngle));
        else
            % Ramp down: 180 -> 90
            alpha = 180 - (180 - 90) * ((theta - midAngle) / (endAngle - midAngle));
        end
    end
end

%% Visualization Function
function visualizeSwingArm(pivotCenter, fieldSize, theta_deg, xTower, yTower, xTip, yTip, alpha_deg)
    % Setup figure
    figure('Color', 'w');
    hold on; axis equal; grid on;
    axis([0 fieldSize 0 fieldSize]);
    title('Swing Arm Movement (Real Data)', 'FontSize', 14);
    xlabel('X (m)', 'FontSize', 12);
    ylabel('Y (m)', 'FontSize', 12);

    % Plot field boundary
    plot([0 fieldSize fieldSize 0 0], [0 0 fieldSize fieldSize 0], 'k-', 'LineWidth', 1.5);

    % Plot trajectory of last tower and tip
    plot(xTower, yTower, 'b--', 'LineWidth', 1.5, 'DisplayName', 'Tower Trajectory');
    plot(xTip, yTip, 'g--', 'LineWidth', 1.5, 'DisplayName', 'Tip Trajectory');

    % Initial tower and tip markers
    towerPlot = plot(xTower(1), yTower(1), 'bo', 'MarkerFaceColor', 'b', 'DisplayName', 'Last Tower');
    tipPlot = plot(xTip(1), yTip(1), 'go', 'MarkerFaceColor', 'g', 'DisplayName', 'Arm Tip');

    % Swing arm line
    armLine = plot([xTower(1), xTip(1)], [yTower(1), yTip(1)], 'g-', 'LineWidth', 2, 'DisplayName', 'Swing Arm');

    % NEW: Blue line connecting pivot center to last tower
    pivotToTowerLine = plot([pivotCenter(1), xTower(1)], ...
                            [pivotCenter(2), yTower(1)], ...
                            'b-', 'LineWidth', 2, 'DisplayName', 'Pivot to Tower');

    % Angle info
    txt = text(50, 950, '', 'FontSize', 12);

    legend('Location', 'best');

    % Animation loop
    for k = 1:length(theta_deg)
        % Update tower and tip positions
        set(towerPlot, 'XData', xTower(k), 'YData', yTower(k));
        set(tipPlot, 'XData', xTip(k), 'YData', yTip(k));

        % Update swing arm line
        set(armLine, 'XData', [xTower(k), xTip(k)], 'YData', [yTower(k), yTip(k)]);

        % Update blue line (pivot to tower)
        set(pivotToTowerLine, 'XData', [pivotCenter(1), xTower(k)], ...
                              'YData', [pivotCenter(2), yTower(k)]);

        % Update angle info
        set(txt, 'String', sprintf('\\theta=%.1f^{\\circ}, \\alpha=%.1f^{\\circ}', theta_deg(k), alpha_deg(k)));

        drawnow;
        pause(0.02);
    end
end
